using System;
using System.Collections;


namespace DarkStrideToolbox
{
	public enum enumType
	{
		Long,
		Int,
		Double,
		String,
		Bool,
		NotSet,
		List,
		Float,
		DoubleZero,
		FloatZero,
		IntZero,
		LongZero,
		Serialization,
		EndOfTypes = 15
	}


	public class DSSerialize : IEnumerable, IEnumerator
	{
		#region Properties
		private const int m_cTYPESTORAGE_NUMBITS = 4;
		//05/02/2007 Chris Hill  Decreased to 7 from 6
		private const int m_cINTLONG_INTSIZESTORAGE_NUMOFBITS = 6;
		//05/02/2007 Chris Hill  Decreased to 7 from 5
		private const int m_cDOUBLE_DECPRECISION_NUMOFPLACES = 5;
		//05/02/2007 Chris Hill  Decreased to 7 from 6
		private const int m_cDOUBLE_INTSIZESTORAGE_NUMOFBITS = 6;
		//05/02/2007 Chris Hill  Decreased to 7 from 5
		private const int m_cDOUBLE_DECSIZESTORAGE_NUMOFBITS = 5;		
		//05/02/2007 Chris Hill  Decreased to 4 from 5
		private const int m_cSTRING_LENSTORAGE_NUMOFBITS = 4;
		//private const int m_cSTRING_INDEXSIZESTORAGE_NUMOFBITS = 4;
		//private const int m_cSTRING_LISTINDEXSIZESTORAGE_NUMOFBITS = 4;
		private const int m_cHEADER_CLASSVER_SIZESTORAGE_NUMOFBITS = 4;
		private const int m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS = 6;
		//05/02/2007 Chris Hill  Decreased to 10 from 15
		private const int m_cHEADER_NUMELEMENTS_SIZESTORAGE_NUMOFBITS = 10;

		private int m_nVersion = 0;

		private int m_nCurrentEnumeratorIndex = 0;
		private object[] m_oPrepValues = null;
		#endregion


		public DSSerialize()
		{
			m_oPrepValues = new object[0];
		}
		public DSSerialize( int nVersion )
		{
			m_nVersion = nVersion;
			m_oPrepValues = new object[0];
		}
		public DSSerialize( string sSerialzed )
		{
			m_oPrepValues = new object[0];

			DeSerialize( sSerialzed );
		}


		public string Serialize()
		{
			return( Serialize( m_nVersion ) );
		}
		public string Serialize( int nVersion )
		{
			DSBitArray oSerializedValues = null;
			ArrayList oStringValues = null;
			string sValue = "";
			//string sBitArray = "";
			string sRetVal = "";


			SerializeToBitArray( nVersion,ref oSerializedValues,ref oStringValues );

			//Convert us
			sRetVal = DSBitArray.ConvertBitArrayToString( oSerializedValues );

			//Convert the strings
			for( int i=0 ; i<oStringValues.Count ; i++ )
			{
				sValue = (string)oStringValues[ i ];
				sRetVal += sValue;
			}			


			return( sRetVal );
		}
		public byte[] SerializeToByteArray()
		{
			return( SerializeToByteArray( 0 ) );
		}
		public byte[] SerializeToByteArray( int nVersion )
		{
			DSBitArray oSerializedValues = null;
			byte[] oaRetVal = null;
			string sValue = "";
			//byte[] baStringValues = null;
			ArrayList oStringValues = null;
			int[] naValues = null;
			int nPos = 0;
			char[] caValues = null;
			long nTotalLen = 0;


			SerializeToBitArray( nVersion,ref oSerializedValues,ref oStringValues );

			//Get the total length of the strings
			for( int i=0 ; i<oStringValues.Count ; i++ )
			{
				if( oStringValues[ i ] != null )
				{
					nTotalLen += ((string)oStringValues[ i ]).Length;
				}
			}

			//Convert us
			oaRetVal = new byte[ oSerializedValues.Length / 8 + nTotalLen ];
			oSerializedValues.InternalBitArray.CopyTo( oaRetVal,0 );

			//Now add in the strings
			nPos = oSerializedValues.Length / 8;
			for( int i=0 ; i<oStringValues.Count ; i++ )
			{
				sValue = (string)oStringValues[ i ];
				caValues = sValue.ToCharArray();
				naValues = new int[ sValue.Length ];
				//baStringValues = new byte[ saStringValues[ i ].Length ];

				caValues.CopyTo( naValues,0 );
				
				for( int nIndex=0 ; nIndex<naValues.Length ; nIndex++ )
				{
					oaRetVal[ nPos + nIndex ] = (byte)naValues[ nIndex ];
				}
				nPos += naValues.Length;
			}


			return( oaRetVal );
		}
		private void SerializeToBitArray( int nVersion,ref DSBitArray oSerializedValues,ref ArrayList oStringValues )
		{
			enumType nEnumType = enumType.NotSet;
			//ArrayList oList = null;
			ArrayList naStringsToAddIn = new ArrayList();
			//int nNextStringIndexAvailable = 0;
			int nPadLen = 0;
			int nPos = 0;


			//Make our array
			oSerializedValues = new DSBitArray( m_oPrepValues.Length * 25 );
			oStringValues = new ArrayList();
			nPos = 0;

			//We store a version for this class to
			//The first bit is an extendor bit... its always zero for now, but in the future if I exceed my storage, 
			//then it will say the next number of bits needed.
			DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cHEADER_CLASSVER_SIZESTORAGE_NUMOFBITS,1 );
			nPos += m_cHEADER_CLASSVER_SIZESTORAGE_NUMOFBITS;

			//and a version for this map
			if( nVersion > Math.Pow( 2,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS ) )
			{
				throw new System.Exception( "You are trying to save the version number " + nVersion + ", however only " +
							m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS.ToString() + " bits have been allocated for this field so " +
							Math.Pow( 2,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS ).ToString() + " is the biggest number you can use." );
			}
			DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS,nVersion );
			nPos += m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS;


			//Now fill our array
			for( int i=0 ; i<m_oPrepValues.Length ; i++ )
			{
				//Data type (Int, Double, String, Bool, NotSet)
				nEnumType = GetTypeEnum( m_oPrepValues[i] );
				StoreVariable( m_oPrepValues[i],ref oSerializedValues,ref nPos,ref oStringValues );
			}


			//Tag that the next group is all strings
			DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cTYPESTORAGE_NUMBITS,Convert.ToInt32( enumType.EndOfTypes ) );
			nPos += m_cTYPESTORAGE_NUMBITS;
 

			//Now that we know how much space we need, we must handle the roll-over.  By roll over I mean, 
			//what if we end up with only 7 bites.  It takes 8 bits to make a char, so we just concatinate
			//a zero on the end.  But now how does the deserialization process know if that zero is valid
			//or not?  Each field can be a minimum of 3 bits, which means that the system may think there
			//is a field on the end when there isn't really.  So... we will pad the front with zeros and
			//then a 1.  THEN the version.  So in a perfect world there will be one wasted bit at the,
			//beginning, but could be as many as 8.
			nPadLen = ( 8 - ( oSerializedValues.Length % 8 ) );
			oSerializedValues.Insert( 0,nPadLen );
			oSerializedValues.Set( nPadLen-1,true );
			nPos += nPadLen;
		}
		private void StoreVariable( object oVariable,ref DSBitArray oSerializedValues,ref int nPos,ref ArrayList oStringValues )
		{
			ArrayList oList = null;
			enumType nEnumType = enumType.NotSet;
			string sValue = "";
			long nIntValue = 0;
			long nIntDecValue = 0;
			double nDecValue = 0;
			double dValue = 0;
			bool bValue = false;


			//Data type (Int, Double, String, Bool, NotSet)
			nEnumType = GetTypeEnum( oVariable );
			DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cTYPESTORAGE_NUMBITS,Convert.ToInt32( nEnumType ) );
			nPos += m_cTYPESTORAGE_NUMBITS;

			if( nEnumType == enumType.NotSet )
			{
				//This means there is a gap, so make sure it gets filled
			}
			else if( nEnumType == enumType.LongZero || nEnumType == enumType.IntZero ||
					 nEnumType == enumType.DoubleZero || nEnumType == enumType.FloatZero )
			{
				//This is to save some space.  If you transfer a zero in a double format, your sending 14 
				//bytes of information when you really need to send zero.  So instead, send zero.
			}
			else if( nEnumType == enumType.List || nEnumType == enumType.Serialization )
			{
				if( nEnumType == enumType.List )
				{
					oList = (ArrayList)oVariable;
				}
				else if( nEnumType == enumType.Serialization )
				{
					oList = ((DSSerialize)oVariable).GetValuesAsArrayList();

					//Do we have a version?
					if( m_nVersion > 0 )
					{
						//1 = yes, we have a version
						DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,1,1 );
						nPos += 1;

						//and a version for this map
						if( m_nVersion > Math.Pow( 2,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS ) )
						{
							throw new System.Exception( "You are trying to save the version number " + m_nVersion + ", however only " +
								m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS.ToString() + " bits have been allocated for this field so " +
								Math.Pow( 2,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS ).ToString() + " is the biggest number you can use." );
						}
						DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS,m_nVersion );
						nPos += m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS;
					}
					else
					{
						//0 = no, we have a version
						DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,1,0 );
						nPos += 1;
					}
				}

				//Now store how many items we have
				DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,m_cHEADER_NUMELEMENTS_SIZESTORAGE_NUMOFBITS,(int)oList.Count );
				nPos += m_cHEADER_NUMELEMENTS_SIZESTORAGE_NUMOFBITS;

				//Now store the items
				for( int nListItemIndex=0 ; nListItemIndex<oList.Count ; nListItemIndex++ )
				{
					StoreVariable( oList[ nListItemIndex ],ref oSerializedValues,ref nPos,ref oStringValues );
				}
			}
			else if( nEnumType == enumType.Long || nEnumType == enumType.Int )
			{
				nIntValue = System.Convert.ToInt32( oVariable );

				//Negative or not
				oSerializedValues.Set( nPos , ( nIntValue < 0 ) );
				nPos += 1;

				StoreDynamicLong( nIntValue,m_cINTLONG_INTSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
			}
			else if( nEnumType == enumType.Double || nEnumType == enumType.Float )
			{
				//Get our int value
				dValue = System.Convert.ToDouble( oVariable );
				if( dValue < 0 )
				{
					nIntValue = (long)Math.Ceiling( dValue );
				}
				else
				{
					nIntValue = (long)Math.Floor( dValue );
				}
				//Get our dec value
				nDecValue = dValue - nIntValue;
				nIntDecValue = (long)Math.Abs( nDecValue * Math.Pow( 10,m_cDOUBLE_DECPRECISION_NUMOFPLACES ) );

				//Negative or not
				oSerializedValues.Set( nPos , ( dValue < 0 ) );
				nPos += 1;

				//Store our int value
				StoreDynamicLong( nIntValue,m_cDOUBLE_INTSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );

				//Store our dec value
				StoreDynamicLong( nIntDecValue,m_cDOUBLE_DECSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
			}
				//Not currently used.
			else if( nEnumType == enumType.String )
			{
				sValue = System.Convert.ToString( oVariable );

				//Store the length
				StoreDynamicLong( sValue.Length,m_cSTRING_LENSTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );

				//Store the value
				oStringValues.Add( sValue );
				
				//Store the index
				//StoreDynamicLong( nIndex,m_cSTRING_INDEXSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );

				//Store the sub index
				//StoreDynamicLong( nSubIndex,m_cSTRING_LISTINDEXSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
			}
			else if( nEnumType == enumType.Bool )
			{
				//Bools are easy
				bValue = System.Convert.ToBoolean( oVariable );
				oSerializedValues.Set( nPos , bValue );
				nPos += 1;
			}
			else
			{
				throw new System.Exception( "Data type not known." );
			}
		}
		private void StoreDynamicLong( long nVariable,int nFixedStorageSize,ref DSBitArray oSerializedValues,ref int nPos )
		{
			int nDidgets = 0;


			try
			{
				//Calculate how many didgets needed
				nDidgets = DSBitArray.GetNumberOfBitsUsedIn( Math.Abs( nVariable ) );

				//Size of storage in bytes
				DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,nFixedStorageSize,nDidgets );
				nPos += nFixedStorageSize;

				//Store the actual value
				DSBitArray.ConvertIntToBitArray( oSerializedValues,nPos,nDidgets,Math.Abs( nVariable ) );
				nPos += nDidgets;
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}
		}


		public void DeSerialize( string sSerialized )
		{
			DeSerialize( sSerialized,ref m_nVersion );
		}
		public void DeSerialize( string sSerialized,ref int nVersion )
		{
			DeSerializeFromBitArray( sSerialized,ref nVersion );
		}
		public void DeSerializeFromByteArray( byte[] oaSerialized )
		{
			int nVersion = 0;

			DeSerializeFromByteArray( oaSerialized,ref nVersion );
		}
		public void DeSerializeFromByteArray( byte[] oaSerialized,ref int nVersion )
		{
			DeSerializeFromBitArray( oaSerialized,ref nVersion );
		}
		private void DeSerializeFromBitArray( object oSourceData,ref int nVersion )
		{
			enumType nEnumType = enumType.NotSet;
			//ArrayList oList = null;
			ArrayList oObjects = new ArrayList();
			ArrayList oStringObjects = new ArrayList();
			//StringData oLoopStringData = null;
			DSBitArray oSerializedValues = null;
			//string sTemp = "";
			object oValue = null;
			object oSerObj = null;
			long nSerializingVersion = 0;
			long nDataType = 0;
			//int nListCount = 0;
			int nStartingStringChar = 0;
			int nPos = 0;
			int nItemIndex = 0;
			int nClassVersion = 0;


			//Setup our source entity.  All that means is the bits will be copied over as needed instead of all at 
			//once.  This is done to speed things up.
			oSerializedValues = new DSBitArray( 0 );
			oSerializedValues.SourceEntity = oSourceData;

			//Now that we know how much space we need, we must handle the roll-over.  By roll over I mean, 
			//what if we end up with only 7 bites.  It takes 8 bits to make a char, so we just concatinate
			//a zero on the end.  But now how does the deserialization process know if that zero is valid
			//or not?  Each field can be a minimum of 3 bits, which means that the system may think there
			//is a field on the end when there isn't really.  So... we will pad the front with zeros and
			//then a 1.  THEN the version.  So in a perfect world there will be one wasted bit at the,
			//beginning, but could be as many as 8.  
			//Start by finding the first 1, thats our real start
			while( nPos<8 && oSerializedValues.Get( nPos ) == false )
			{
				nPos++;
			}
			nPos++;


			//nVersion (0-63)
			nSerializingVersion = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cHEADER_CLASSVER_SIZESTORAGE_NUMOFBITS );
			nPos += m_cHEADER_CLASSVER_SIZESTORAGE_NUMOFBITS;
			if( nSerializingVersion != 1 )
			{
				throw new System.Exception( "Unable to deserialize this object, it was serialized from version " + nSerializingVersion.ToString() + ", which is not supported." );
			}

			//nVersion (0-63)
			nClassVersion = (int)DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS );
			m_nVersion = nClassVersion;
			nVersion = m_nVersion;
			nPos += m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS;


			while( nPos<oSerializedValues.Length )
			{
				//Data type (Int, Double, String, Bool, NotSet)
				nDataType = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cTYPESTORAGE_NUMBITS );
				nEnumType = (enumType)nDataType;
				//Intentionally don't incrament, retreive variable does that
				//nPos += m_cTYPESTORAGE_NUMBITS;

				if( nEnumType == enumType.EndOfTypes )
				{
					//Ok we want to consume the type storage
					nPos += m_cTYPESTORAGE_NUMBITS;
					break;
				}
				else
				{
					oValue = RetreiveVariable( ref oSerializedValues,ref nPos,ref oStringObjects );
				}

				nItemIndex++;

				oObjects.Add( oValue );
			}


			//Copy the values over
			m_oPrepValues = new object[ oObjects.Count ];
			for( int i=0 ; i<oObjects.Count ; i++ )
			{
				m_oPrepValues[ i ] = oObjects[ i ];
			}


			//Now handle our string data
			//oList = GetValuesAsArrayList();
			nStartingStringChar = (int)Math.Floor( (double)nPos / 8.0 );
			oSerObj = this;
			PassOutStringData( oSourceData,ref oSerObj,ref nStartingStringChar );
		}

		private void PassOutStringData( object oSourceData,ref object oObject,ref int nPosStringsStartAt )
		{
			ArrayList oList = null;
			StringData oLoopStringData = null;
			DSSerialize oSer = null;
			string sStringValue = "";
			object oObj = null;


			//Walk through all our data looking for string peices
			if( GetTypeEnum( oObject ) == enumType.List )
			{
				oList = (ArrayList)oObject;

				for( int i=0 ; i<oList.Count ; i++ )
				{
					oObj = oList[ i ];
					if( oObj != null )
					{
						PassOutStringData( oSourceData,ref oObj,ref nPosStringsStartAt );
						oList[ i ] = oObj;
					}
				}
			}
			else if( GetTypeEnum( oObject ) == enumType.Serialization )
			{
				oSer = (DSSerialize)oObject;

				for( int i=0 ; i<oSer.NumberOfParamaters ; i++ )
				{
					oObj = oSer.Get( i );
					if( oObj != null )
					{
						PassOutStringData( oSourceData,ref oObj,ref nPosStringsStartAt );
						oSer.Set( i,oObj );
					}
				}
			}
			else if( oObject.GetType() == typeof( StringData ) )
			{
				oLoopStringData = (StringData)oObject;

				//Ok where does this object go?
				if( oSourceData.GetType() == typeof( string ) )
				{
					string sValue = (string)oSourceData;
					sStringValue = sValue.Substring( nPosStringsStartAt,(int)oLoopStringData.m_nStrLen );
				}
				else if( oSourceData.GetType() == typeof( byte[] ) )
				{
					//Convert to an array of chars
					char[] caValues = new char[ oLoopStringData.m_nStrLen ];
					byte[] oaBytes = (byte[])oSourceData;
					for( int nStrIndx=0 ; nStrIndx<oLoopStringData.m_nStrLen ; nStrIndx++ )
					{
						caValues[ nStrIndx ] = (char)oaBytes[ nPosStringsStartAt + nStrIndx ];
					}
					//Convert to a string
					sStringValue = new string( caValues );
				}


				//Advance our tracking
				nPosStringsStartAt += (int)oLoopStringData.m_nStrLen;
				oObject = sStringValue;
			}
		}

		private object RetreiveVariable( ref DSBitArray oSerializedValues,ref int nPos,ref ArrayList oStringObjects )
		{
			enumType nEnumType = enumType.NotSet;
			object oValue = null;
			long nValue = 0;
			long nNegative = 0;
			long nDataType = 0;
			//long nStrLen = 0;
			//long nIndex = 0;
			long nListCount = 0;
			//long nSubIndex = 0;
			double dIntValue = 0;
			double dDecValue = 0;
			StringData oStringData = null;			
			ArrayList oList = null;
			ArrayList oObjects = new ArrayList();
			DSSerialize oSer = null;


			//Data type (Int, Double, String, Bool, NotSet)
			nDataType = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cTYPESTORAGE_NUMBITS );
			nEnumType = (enumType)nDataType;
			nPos += m_cTYPESTORAGE_NUMBITS;

			//Now the actual value
			if( nEnumType == enumType.LongZero )
			{
				oValue = (long)0;
			}
			else if( nEnumType == enumType.IntZero )
			{
				oValue = (int)0;
			}
			else if( nEnumType == enumType.DoubleZero )
			{
				oValue = (double)0;
			}
			else if( nEnumType == enumType.FloatZero )
			{
				oValue = (float)0;
			}
			else if( nEnumType == enumType.List || nEnumType == enumType.Serialization )
			{
				if( nEnumType == enumType.Serialization )
				{
					nValue = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,1 );
					nPos++;

					//Do we have a version?
					if( nValue == 1 )
					{
						//And a version for this map
						m_nVersion = (int)DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS );
						nPos += m_cHEADER_USERVER_SIZESTORAGE_NUMOFBITS;
					}
				}


				//Now find out how many items we have
				nListCount = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cHEADER_NUMELEMENTS_SIZESTORAGE_NUMOFBITS );
				nPos += m_cHEADER_NUMELEMENTS_SIZESTORAGE_NUMOFBITS;

				//Get all those items
				oList = new ArrayList();
				oSer = new DSSerialize();
				for( int nListItemIndex=0 ; nListItemIndex<nListCount ; nListItemIndex++ )
				{
					oValue = RetreiveVariable( ref oSerializedValues,ref nPos,ref oStringObjects );

					if( nEnumType == enumType.List )
					{				
						oList.Add( oValue );
					}
					else if( nEnumType == enumType.Serialization )
					{
						oSer.Set( nListItemIndex,oValue );
					}					
				}

				//Now return our list
				if( nEnumType == enumType.List )
				{				
					oValue = oList;
				}
				else if( nEnumType == enumType.Serialization )
				{
					oValue = oSer;
				}
			}
			else if( nEnumType == enumType.Long || nEnumType == enumType.Int )
			{
				//Negative or not
				nNegative = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,1 );
				nPos += 1;

				oValue = RetreiveDynamicInt( m_cINTLONG_INTSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
				/*//Size in bytes
				nNumeratorSize = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cINTLONG_INTSIZESTORAGE_NUMOFBITS );
				nPos += m_cINTLONG_INTSIZESTORAGE_NUMOFBITS;
				//Calculate the actual value
				oValue = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,nNumeratorSize );*/
				if( nEnumType == enumType.Int )
				{
					oValue = Convert.ToInt32( oValue );
				}
				else
				{
					oValue = Convert.ToInt64( oValue );
				}

				//Negativity
				if( nNegative == 1 )
				{
					if( nEnumType == enumType.Int )
					{
						oValue = Convert.ToInt32( oValue ) * -1;
					}
					else
					{
						oValue = Convert.ToInt64( oValue ) * -1;
					}
				}
			}
			else if( nEnumType == enumType.Double || nEnumType == enumType.Float )
			{
				//Negative or not
				nNegative = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,1 );
				nPos += 1;

				//Retreive the int value
				oValue = RetreiveDynamicInt( m_cDOUBLE_INTSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
				dIntValue = Convert.ToDouble( oValue );
				/*//Size in bytes
				nNumeratorSize = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cDOUBLE_INTSIZESTORAGE_NUMOFBITS );
				nPos += m_cDOUBLE_INTSIZESTORAGE_NUMOFBITS;
				//Calculate the actual value
				oValue = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,nNumeratorSize );
				dIntValue = Convert.ToDouble( oValue );
				nPos += (int)nNumeratorSize;*/

				//Retreive the dec value
				oValue = RetreiveDynamicInt( m_cDOUBLE_DECSIZESTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );
				dDecValue = Convert.ToDouble( oValue );
				/*//Size in bytes
				nDenomSize = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,m_cDOUBLE_DECSIZESTORAGE_NUMOFBITS );
				nPos += m_cDOUBLE_DECSIZESTORAGE_NUMOFBITS;
				//Calculate the actual value
				oValue = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,nDenomSize );
				dDecValue = Convert.ToDouble( oValue );
				nPos += (int)nDenomSize;*/

				//Merge the num and denom.
				dDecValue = dDecValue / Math.Pow( 10,m_cDOUBLE_DECPRECISION_NUMOFPLACES );
				oValue = dIntValue + dDecValue;
				if( nEnumType == enumType.Double )
				{
					oValue = Convert.ToDouble( oValue );
				}
				else
				{
					oValue = (float)Convert.ToDouble( oValue );
				}

				//Negativity
				if( nNegative == 1 )
				{
					if( nEnumType == enumType.Double )
					{
						oValue = Convert.ToDouble( oValue ) * -1;
					}
					else
					{
						oValue = (float)Convert.ToDouble( oValue ) * -1;
					}
				}
			}
				//Not currently used.
			else if( nEnumType == enumType.String )
			{
				oStringData = new StringData();

				//Get the string length
				oStringData.m_nStrLen = RetreiveDynamicInt( m_cSTRING_LENSTORAGE_NUMOFBITS,ref oSerializedValues,ref nPos );

				oValue = oStringData;
			}
			else if( nEnumType == enumType.Bool )
			{
				//Bools are easy
				oValue = oSerializedValues.Get( nPos );
				nPos += 1;					
			}
				//This means there is a gap
			else if( nEnumType == enumType.NotSet )
			{
				oValue = null;
			}


			return( oValue );
		}
		private long RetreiveDynamicInt( int nFixedStorage,ref DSBitArray oSerializedValues,ref int nPos )
		{
			long nDidgets = 0;
			long nRetVal = 0;


			//Size in bytes
			nDidgets = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,nFixedStorage );
			nPos += nFixedStorage;

			//Calculate the actual value
			nRetVal = DSBitArray.ConvertBitArrayToInt( oSerializedValues,nPos,nDidgets );
			nPos += (int)nDidgets;


			return( nRetVal );
		}

        
		#region Enumerator Functions
		public IEnumerator GetEnumerator() 
		{ 
			return( (IEnumerator)this ); 
		} 

		public bool MoveNext() 
		{ 
			if( m_nCurrentEnumeratorIndex < m_oPrepValues.Length ) 
			{ 
				m_nCurrentEnumeratorIndex++; 
				return true; 
			} 
			else 
			{
				return( false ); 
			} 
		} 

		public object Current 
		{ 
			get 
			{ 
				return( m_oPrepValues[ m_nCurrentEnumeratorIndex ] ); 
			} 
		} 

		public void Reset() 
		{ 
			m_nCurrentEnumeratorIndex = 0;
		} 

		#endregion

		public void Set( int nIndex,object oValue )
		{
			object[] oNewStorage = null;

			if( nIndex >= m_oPrepValues.Length )
			{
				oNewStorage = new object[ nIndex + 1 ];

				for( int i=0 ; i<m_oPrepValues.Length ; i++ )
				{
					oNewStorage[ i ] = m_oPrepValues[ i ];
				}

				m_oPrepValues = oNewStorage;
			}

			m_oPrepValues[ nIndex ] = oValue;
		}
		public void SetList( int nIndex )
		{
			Set( nIndex,new ArrayList() );
		}
		public void SetListItem( int nIndex,object oValue )
		{
			object oListObj = null;
			ArrayList oArrayList =null;


			//Verify that this item is a list
			oListObj = Get( nIndex );
			if( GetTypeEnum( oListObj ) != enumType.List )
			{
				throw new System.Exception( "Item " + nIndex.ToString() + " is not a list." );
			}

			//Get sorted list
			oArrayList = (ArrayList)oListObj;

			//Append item
			oArrayList.Add( oValue );

			//Save it back
			Set( nIndex,oArrayList );
		}

		#region Get Functions
		public object Get( int nIndex )
		{
			return( Get( nIndex,null ) );
		}
		public object Get( int nIndex,object oDefaultIfNotFound )
		{
			if( nIndex < 0 ||
				nIndex >= m_oPrepValues.Length ||
				m_oPrepValues[ nIndex ] == null )
			{
				return( oDefaultIfNotFound );
			}
			else
			{
				return( m_oPrepValues[ nIndex ] );
			}
		}

		public int GetInt( int nIndex )
		{
			return( GetInt( nIndex,0 ) );
		}
		public int GetInt( int nIndex,int nDefaultIfNotFound )
		{
			return( (int)Get( nIndex,nDefaultIfNotFound ) );
		}
		public string GetString( int nIndex )
		{
			return( (string)Get( nIndex,"" ) );
		}
		public string GetString( int nIndex,string sDefaultIfNotFound )
		{
			return( (string)Get( nIndex,sDefaultIfNotFound ) );
		}
		public double GetDouble( int nIndex )
		{
			return( (double)Get( nIndex,(double)0 ) );
		}
		public double GetDouble( int nIndex,double nDefaultIfNotFound )
		{
			return( (double)Get( nIndex,nDefaultIfNotFound ) );
		}
		public float GetFloat( int nIndex )
		{
			return( (float)Get( nIndex,(float)0 ) );
		}
		public float GetFloat( int nIndex,float nDefaultIfNotFound )
		{
			return( (float)Get( nIndex,nDefaultIfNotFound ) );
		}
		public long GetLong( int nIndex )
		{
			return( (long)Get( nIndex,(long)0 ) );
		}
		public long GetLong( int nIndex,long nDefaultIfNotFound )
		{
			return( (long)Get( nIndex,nDefaultIfNotFound ) );
		}
		public bool GetBool( int nIndex )
		{
			return( (bool)Get( nIndex,false ) );
		}
		public bool GetBool( int nIndex,bool bDefaultIfNotFound )
		{
			return( (bool)Get( nIndex,bDefaultIfNotFound ) );
		}
		public ArrayList GetList( int nIndex )
		{
			object oListObj = null;
			ArrayList oArrayList =null;


			//Verify that this item is a list
			oListObj = Get( nIndex );
			if( GetTypeEnum( oListObj ) != enumType.List )
			{
				throw new System.Exception( "Item " + nIndex.ToString() + " is not a list." );
			}

			//Get sorted list
			oArrayList = (ArrayList)oListObj;

			
			return( oArrayList );
		}
		#endregion
		public ArrayList GetValuesAsArrayList()
		{
			ArrayList oList = new ArrayList();


			for( int i=0 ; i<this.NumberOfParamaters ; i++ )
			{
				oList.Add(  this.Get( i ) );
			}


			return( oList );
		}


		public enumType GetTypeEnum( int nIndex )
		{
			object oValue = null;

			oValue = this.Get( nIndex );

			return( GetTypeEnum( oValue ) );
		}
		public static enumType GetTypeEnum( object oValue )
		{
			enumType nRetVal = enumType.NotSet;

			if( oValue == null )	
			{ 
				nRetVal = enumType.NotSet; 
			}
			else
			{
				if( oValue.GetType() == typeof( DSSerialize ) )
				{
					nRetVal = enumType.Serialization;
				}
				else if( oValue.GetType() == typeof( long ) )		
				{ 
					if( (long)oValue == 0 )
					{
						nRetVal = enumType.LongZero; 
					}
					else
					{
						nRetVal = enumType.Long; 
					}
				}
				else if( oValue.GetType() == typeof( int ) )		
				{ 
					if( (int)oValue == 0 )
					{
						nRetVal = enumType.IntZero; 
					}
					else
					{
						nRetVal = enumType.Int; 
					}
				}
				else if( oValue.GetType() == typeof( double ) )	
				{ 
					if( (double)oValue == 0 )
					{
						nRetVal = enumType.DoubleZero; 
					}
					else
					{
						nRetVal = enumType.Double; 
					}
				}
				else if( oValue.GetType() == typeof( string ) )		{ nRetVal = enumType.String; }
				else if( oValue.GetType() == typeof( bool ) )		{ nRetVal = enumType.Bool; }
				else if( oValue.GetType() == typeof( ArrayList ) )	{ nRetVal = enumType.List; }
				else if( oValue.GetType() == typeof( float ) )		
				{
					if( (float)oValue == 0 )
					{
						nRetVal = enumType.FloatZero; 
					}
					else
					{ 
						nRetVal = enumType.Float; 
					}
				}
			}

			return( nRetVal );
		}


		public string PerformSpeedAndSizeTests( int nNumTestsToRun )
		{
			string sResults = "";
			string sSer1 = "";
			string sSer2 = "";
			string sSerText = "";
			string sDeSerText = "";
			string sError = "";
			int nNumTests_String = 100 + nNumTestsToRun;//10000
			int nNumTests_Mixed = 100 + nNumTestsToRun;//2000
			DSSerialize oSer = null;
			DSSerialize oDeSer = null;
			DateTime dtStart1 = DateTime.MinValue;
			DateTime dtStop1 = DateTime.MinValue;
			DateTime dtStart2 = DateTime.MinValue;
			DateTime dtStop2 = DateTime.MinValue;
			TimeSpan oSpan1 = TimeSpan.MinValue;
			TimeSpan oSpan2 = TimeSpan.MinValue;
			string[] saSplit = null;


			sResults = "Test                       Old Time       New Time         Old Size      New Size";
			sResults += "\n----------------------------------------------------------------------------------";

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			// String Serialize Tests
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			for( int nNumLoops=1 ; nNumLoops < 4000 ; nNumLoops *= 2 )
			{
				//First of all, strings alone, in this method
				oSer = new DSSerialize();
				dtStart1 = DateTime.Now;
				for( int i=0 ; i<nNumTests_String ; i++ )
				{
					for( int j=0 ; j<nNumLoops ; j++ )
					{
						oSer.Set( j,"Whats up buddy?!" );
					}
				}
				sSer1 = oSer.Serialize();
				dtStop1 = DateTime.Now;
				oSpan1 = dtStop1 - dtStart1;

				//First of all, strings alone, normal method
				dtStart2 = DateTime.Now;
				for( int i=0 ; i<nNumTests_String ; i++ )
				{
					sSer2 = "";
					for( int j=0 ; j<nNumLoops ; j++ )
					{
						sSer2 += "[SEP]Whats up buddy?!";
					}
				}
				dtStop2 = DateTime.Now;
				oSpan2 = dtStop2 - dtStart2;
				sSerText += "\n" + nNumLoops.ToString( "0000" ) + " Ser Strings:      " + 
								oSpan2.TotalSeconds.ToString( "00.0000" ) + "       " + 
								oSpan1.TotalSeconds.ToString( "00.0000" ) + "           " + 
								sSer2.Length.ToString( "00000" ) + "        " + 
								sSer1.Length.ToString( "00000" ) + "       ";

				//////////////////////////////////////////////////////////////////////////////////////////////////////
				// String DeSerialize Tests
				//////////////////////////////////////////////////////////////////////////////////////////////////////
				//First of all, strings alone, in this method
				oDeSer = new DSSerialize();
				dtStart1 = DateTime.Now;
				for( int i=0 ; i<nNumTests_String ; i++ )
				{
					oDeSer.DeSerialize( sSer1 );
				}
				dtStop1 = DateTime.Now;
				oSpan1 = dtStop1 - dtStart1;
				//Verify this last one worked right
				sError = "";
				for( int j=0 ; j<nNumLoops ; j++ )
				{
					if( oSer.GetString( j ) != oDeSer.GetString( j ) )
					{
						sError = "    - Error on " + j.ToString();
					}
				}


				//First of all, strings alone, normal method
				dtStart2 = DateTime.Now;
				for( int i=0 ; i<nNumTests_String ; i++ )
				{
					saSplit = DSMisc.Split( sSer2,"[SEP]" );
				}
				dtStop2 = DateTime.Now;
				oSpan2 = dtStop2 - dtStart2;
				sDeSerText += "\n" + nNumLoops.ToString( "0000" ) + " DeS Strings:      " + 
								oSpan2.TotalSeconds.ToString( "00.0000" ) + "       " + 
								oSpan1.TotalSeconds.ToString( "00.0000" ) + sError;
			}

			sResults += sSerText + "\n----------------------------------------------------------------------------------" +
						sDeSerText + "\n----------------------------------------------------------------------------------";
			sSerText = "";
			sDeSerText = "";

			//////////////////////////////////////////////////////////////////////////////////////////////////////
			// Mixed Ser Tests
			//////////////////////////////////////////////////////////////////////////////////////////////////////
			for( int nNumLoops=1 ; nNumLoops < 400 ; nNumLoops *= 2 )
			{
				//First of all, values alone, in this method
				oSer = new DSSerialize();
				dtStart1 = DateTime.Now;
				for( int i=0 ; i<nNumTests_Mixed ; i++ )
				{
					for( int j=0 ; j<nNumLoops ; j++ )
					{
						oSer.Set( 0+j*12,"Hello" );
						oSer.Set( 1+j*12,(float)1.5 );
						oSer.Set( 2+j*12,(double)2.75 );
						oSer.Set( 3+j*12,true );
						oSer.Set( 4+j*12,(int)3 );
						oSer.Set( 5+j*12,(long)4 );
						oSer.Set( 6+j*12,"Goodbye" );
						oSer.Set( 7+j*12,(float)-5.5 );
						oSer.Set( 8+j*12,(double)-6.75 );
						oSer.Set( 9+j*12,false );
						oSer.Set( 10+j*12,(int)-7 );
						oSer.Set( 11+j*12,(long)-8 );
					}
				}
				sSer1 = oSer.Serialize();
				dtStop1 = DateTime.Now;
				oSpan1 = dtStop1 - dtStart1;

				//First of all, strings alone, normal method
				dtStart2 = DateTime.Now;
				for( int i=0 ; i<nNumTests_Mixed ; i++ )
				{
					sSer2 = "";
					for( int j=0 ; j<nNumLoops ; j++ )
					{
						sSer2 += "[SEP]" + "Hello";
						sSer2 += "[SEP]" + ((float)1.5).ToString();
						sSer2 += "[SEP]" + ((double)2.75).ToString();
						sSer2 += "[SEP]" + true.ToString();
						sSer2 += "[SEP]" + ((int)3).ToString();
						sSer2 += "[SEP]" + ((long)4).ToString();
						sSer2 += "[SEP]" + "Goodbye";
						sSer2 += "[SEP]" + ((float)-5.5).ToString();
						sSer2 += "[SEP]" + ((double)-6.75).ToString();
						sSer2 += "[SEP]" + false.ToString();
						sSer2 += "[SEP]" + ((int)-7).ToString();
						sSer2 += "[SEP]" + ((long)-8).ToString();
					}
				}
				dtStop2 = DateTime.Now;
				oSpan2 = dtStop2 - dtStart2;
				sSerText += "\n" + nNumLoops.ToString( "0000" ) + " Ser Mixed:       " + 
								oSpan2.TotalSeconds.ToString( "00.0000" ) + "       " + 
								oSpan1.TotalSeconds.ToString( "00.0000" ) + "           " + 
								sSer2.Length.ToString( "00000" ) + "        " + 
								sSer1.Length.ToString( "00000" ) + "       ";



				//////////////////////////////////////////////////////////////////////////////////////////////////////
				// Mixed DeSer Tests
				//////////////////////////////////////////////////////////////////////////////////////////////////////
				//First of all, values alone, in this method
				oDeSer = new DSSerialize();
				dtStart1 = DateTime.Now;
				for( int i=0 ; i<nNumTests_Mixed ; i++ )
				{
					oDeSer.DeSerialize( sSer1 );
				}
				dtStop1 = DateTime.Now;
				oSpan1 = dtStop1 - dtStart1;
				//Verify this last one worked right
				sError = "";
				for( int j=0 ; j<nNumLoops ; j++ )
				{
					if( oDeSer.GetString( 0+j*12 ) != oSer.GetString( 0+j*12 ) ||
						oDeSer.GetFloat( 1+j*12 ) != oSer.GetFloat( 1+j*12 ) ||
						oDeSer.GetDouble( 2+j*12 ) != oSer.GetDouble( 2+j*12 ) ||
						oDeSer.GetBool( 3+j*12 ) != oSer.GetBool( 3+j*12 ) ||
						oDeSer.GetInt( 4+j*12 ) != oSer.GetInt( 4+j*12 ) ||
						oDeSer.GetLong( 5+j*12 ) != oSer.GetLong( 5+j*12 ) ||
						oDeSer.GetString( 6+j*12 ) != oSer.GetString( 6+j*12 ) ||
						oDeSer.GetFloat( 7+j*12 ) != oSer.GetFloat( 7+j*12 ) ||
						oDeSer.GetDouble( 8+j*12 ) != oSer.GetDouble( 8+j*12 ) ||
						oDeSer.GetBool( 9+j*12 ) != oSer.GetBool( 9+j*12 ) ||
						oDeSer.GetInt( 10+j*12 ) != oSer.GetInt( 10+j*12 ) ||
						oDeSer.GetLong( 11+j*12 ) != oSer.GetLong( 11+j*12 ) )
					{
						sError = "    - Error on " + j.ToString();
					}
				}

				//First of all, strings alone, normal method
				dtStart2 = DateTime.Now;
				for( int i=0 ; i<nNumTests_Mixed ; i++ )
				{
					saSplit = DSMisc.Split( sSer2,"[SEP]" );
				}
				dtStop2 = DateTime.Now;
				oSpan2 = dtStop2 - dtStart2;
				sDeSerText += "\n" + nNumLoops.ToString( "0000" ) + " DeS Mixed:       " + 
								oSpan2.TotalSeconds.ToString( "00.0000" ) + "       " + 
								oSpan1.TotalSeconds.ToString( "00.0000" ) + sError;
			}

			sResults += sSerText + "\n----------------------------------------------------------------------------------" +
						sDeSerText + "\n----------------------------------------------------------------------------------";


			return( sResults );
		}



		#region Properties
		public long NumberOfParamaters
		{
			get
			{
				return( m_oPrepValues.Length );
			}
		}
		public long Version
		{
			get
			{
				return( m_nVersion );
			}
		}
		#endregion
	}

	internal class StringData
	{
		public long m_nStrLen = 0;
	}
}
