using System;
using Microsoft.DirectX.Direct3D;
using Microsoft.DirectX;

using System.Windows.Forms;


namespace DarkStrideToolbox
{
	public class DSParticleSystem
	{
		#region Properties
        private System.Collections.Generic.List<double> m_oAvgAdvanceTime = new System.Collections.Generic.List<double>();
        private double m_nAverageAdvanceTime = .03;

		private Microsoft.DirectX.Direct3D.Device m_oDirect3DDevice = null;
		private DSSortedList m_oParticleManagers = new DSSortedList();

		//11/20/2004 Chris Hill  This tracks weather or not we are in debug mode.  Note that
		//this is different than compile-debug, this is weather or not we write out a log.
		private bool m_bDebugMode = false;
		private string m_sDebugPath = "";
		#endregion


		public DSParticleSystem( Device oDevice )
		{
			m_oDirect3DDevice = oDevice;
		}
		public void RestoreDeviceObjects( Device oDevice )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.RestoreDeviceObjects";

			try
			{
				for( int i=m_oParticleManagers.Count-1 ; i>=0 ; i--)
				{
					DSParticleManager oManager = (DSParticleManager)m_oParticleManagers.GetByIndex( i );
					oManager.RestoreDeviceObjects( oDevice );
				}

				m_oDirect3DDevice = oDevice;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
        public virtual void Dispose()
        {
            const string sRoutineName = "DarkStrideToolbox.ParticleSystem.Dispose";

            try
            {
                m_oDirect3DDevice = null;

                m_oParticleManagers = null;
            }
            catch (System.Exception oEx)
            {
                throw new System.Exception(sRoutineName + " Failed.", oEx);
            }
        }

		public void Advance( double nElapsedTime )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.Advance";
			DSParticleManager oManager = null;

			try
			{
                m_nAverageAdvanceTime = CalculateOurAverageAdvTime(nElapsedTime);

                for (int i = m_oParticleManagers.Count - 1; i >= 0; i--)
                {
                    oManager = (DSParticleManager)m_oParticleManagers.GetByIndex(i);
                    oManager.Advance(nElapsedTime);

                    if (oManager.RunDown == true && oManager.ParticleCount == 0)
                    {
                        m_oParticleManagers.RemoveAt(i);
                        i--;
                    }
                }
                
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
        private double CalculateOurAverageAdvTime(double nElapsedTime)
        {
            double nAvgTime = 0;


            if (m_oAvgAdvanceTime.Count >= 20)
            {
                foreach (double nLoopTime in m_oAvgAdvanceTime)
                {
                    nAvgTime += nLoopTime;
                }
                nAvgTime /= m_oAvgAdvanceTime.Count;

                m_oAvgAdvanceTime.RemoveAt(0);
            }
            else
            {
                nAvgTime = nElapsedTime;
            }
            m_oAvgAdvanceTime.Add(nElapsedTime);


            return (nAvgTime);
        }
		public void Render( Vector2 oViewUpperLeft )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.Render2D";

			try
			{
				//11/20/2004 Chris Hill  Added debugging.
				DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,m_bDebugMode,2,"Starting ParticleSystem.Render2D" );

				m_oDirect3DDevice.RenderState.Lighting = false;

				for( int i=m_oParticleManagers.Count-1 ; i>=0 ; i--)
				{
					DSParticleManager oManager = (DSParticleManager)m_oParticleManagers.GetByIndex( i );
					oManager.Render( m_oDirect3DDevice,oViewUpperLeft  );
				}

				m_oDirect3DDevice.RenderState.Lighting = true;

				// Reset render states
				m_oDirect3DDevice.RenderState.PointSpriteEnable = false;
				m_oDirect3DDevice.RenderState.PointScaleEnable = false;

				m_oDirect3DDevice.RenderState.ZBufferWriteEnable = true;
				m_oDirect3DDevice.RenderState.AlphaBlendEnable = false;


				//11/20/2004 Chris Hill  Added debugging.
				DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,m_bDebugMode,2,
					"Stoping ParticleSystem.Render2D" );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		public DSParticleManager GetParticleManager( string sGUID )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.GetParticleManager";
			DSParticleManager oManager = null;
			bool bFound = false;

			try
			{
				for( int i=0 ; i<m_oParticleManagers.Count ; i++ )
				{
					oManager = (DSParticleManager)m_oParticleManagers.GetByIndex(i);
					if( oManager.GUID.Equals( sGUID ) == true )
					{
						bFound = true;
						break;
					}
				}

				if( bFound == false )
				{
					oManager = null;
				}
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
			return( oManager );
		}

		public DSParticleManager CreateParticleSystem()
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.CreateParticleSystem";
			DSParticleManager oNewManager = null;

			try
			{
				oNewManager = new DSParticleManager( m_oDirect3DDevice );
				m_oParticleManagers.Add( oNewManager.GUID,oNewManager );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
			return( oNewManager );
		}
		public DSParticleManager CreateParticleSystem( System.Drawing.Color[] oaColors,double[] naPercTimeInColor, 
														float fEmitVel,int NumParticlesToEmit, Vector2 vEmitterPosition,
														Vector2 vEmitterDirection )
		{
			DSParticleManager oNewManager = null;

			oNewManager = new DSParticleManager( m_oDirect3DDevice,oaColors,naPercTimeInColor, fEmitVel,NumParticlesToEmit,  vEmitterPosition, vEmitterDirection );
			oNewManager.ParticleSystem = this;
			m_oParticleManagers.Add( oNewManager.GUID,oNewManager );

			return( oNewManager );
		}
		public DSParticleManager CreateParticleSystem( System.Drawing.Color[] oaColors,double[] naPercTimeInColor, 
														float fEmitVel,int NumParticlesToEmit, Vector2 vEmitterPosition,
														Vector2 vEmitterDirection,string sTexture )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.CreateParticleSystem";
			DSParticleManager oNewManager = null;

			try
			{
				oNewManager = new DSParticleManager( m_oDirect3DDevice,oaColors,naPercTimeInColor,fEmitVel, NumParticlesToEmit,  vEmitterPosition, vEmitterDirection );
				oNewManager.ParticleSystem = this;
				oNewManager.ParticleTexture = sTexture;
				m_oParticleManagers.Add( oNewManager.GUID,oNewManager );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
			return( oNewManager );
		}
		public DSParticleManager CreateParticleSystem( DSParticleManager oParticleManager )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleSystem.CreateParticleSystem";

			try
			{
				oParticleManager.ParticleSystem = this;
				oParticleManager.RestoreDeviceObjects( m_oDirect3DDevice );
				m_oParticleManagers.Add( oParticleManager.GUID,oParticleManager );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
			return( oParticleManager );
		}

        public void RemoveParticleManager(string sParticleManagerGuid)
        {
            //DSParticleManager oPMan = null;

            //oPMan = (DSParticleManager)m_oParticleManagers[sParticleManagerGuid];
            m_oParticleManagers.Remove(sParticleManagerGuid);
        }


		#region Properties
		public Device Device
		{
			get
			{
				return( m_oDirect3DDevice );
			}
			set
			{
				m_oDirect3DDevice = value;
			}
		}
		//11/20/2004 Chris Hill  Added a debug mode for tracking a time problem.
		public bool DebugMode
		{
			set
			{
				DSParticleManager oManager = null;


				//Are we turning it on when it was off?
				if( m_bDebugMode == false && value == true )
				{
					//Only make a new log file name if we don't have one already
					if( m_sDebugPath.Length == 0 )
					{
						m_sDebugPath =	DSMisc.GetDevelopmentAppPath() + 
							"DSParticleSystem_DebugFile_" + 
							DateTime.Now.ToString( "yyyy-MM-dd hh" ) + "h" + 
							DateTime.Now.ToString( "mm" ) + "m" + 
							DateTime.Now.ToString( "ss" ) + "s" + 
							".OUT";

						//Then write a header
						DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,value,false,0,
							"\r\n\r\nStarting Debug Log File For DSParticleSystem\r\n" +
							"--------------------------------------------" );
					}
				}
					//Are we turning it off when it was currently on?  Then clear the path.
				else if( m_bDebugMode == true && value == false )
				{
					m_sDebugPath = "";
				}

				m_bDebugMode = value;


				//Now turn it on for all of our child objects
				for( int i=0 ; i<m_oParticleManagers.Count ; i++ )
				{
					oManager = (DSParticleManager)m_oParticleManagers.GetByIndex( i );
					oManager.DebugPath = m_sDebugPath;
					oManager.DebugMode = value;
				}
			}
			get
			{
				return( m_bDebugMode );
			}
		}
		public string DebugPath
		{
			get
			{
				return( m_sDebugPath );
			}
			set
			{
				m_sDebugPath = value;

				//We turn off then on the debug mode so that it writes a new header to the log
				if( m_bDebugMode == true )
				{
					DebugMode = false;
					DebugMode = true;
				}				
			}
		}
        public double AverageAdvanceTime
        {
            get
            {
                return( m_nAverageAdvanceTime );
            }
        }
		#endregion
	}

	public class DSParticleManager
	{
		#region Properties
		//Movement Properties
		private Vector2 m_vGravity = new Vector2( 0,0 );
		private Vector2 m_vEmitterPosition = new Vector2( 0,0 );
		private Vector2 m_vEmitterDirection = new Vector2( 0,0 );
		private Vector2 m_vEmittersVelocity = new Vector2( 0,0 );
		private double m_nEmitterRangeInRad = Math.PI;
		//Particle Properties
		private double m_dParticleLifeTimeInSeconds = 0;
		//General Properties
		private string m_sGUID = "";
		private double m_dCurrentTime = 0.0f;
		//Appearance Properties
		private float m_fParticleSize = 0.8f;

        //How long does the manager live
        private double m_dParticleManagerLifeTimeInSeconds = 0;
        private double m_dParticleManagerTimeAlive = 0;

		private bool m_bEmitParticles = true;
		private bool m_bRunDown = false;
		//11/13/2003 Chris Hill  If its a one shot PM then after its first shot at emitting it switches to a run down PM.
		private bool m_bOneShot = false;

		private float m_fEmitVel = 0;
		private double m_nNumParticlesToEmit = 0;
		private double m_nNumPartialParticlesEmmitedLastClick = 0;

		//11/30/2004 Chris Hill  This is the absolute max number of m_nNumParticles to emit.  The previous variable is how many to
		//emit per second.  After this many m_nNumParticles is admitted the PM goes into run-down mode automatically.  If this is a
		//zero then their is no limit to the number of m_nNumParticles emmited.
		private long m_nMaxNumParticlesToEmit = 0;
		private long m_nNumParticlesEmmited = 0;
	
		public struct PointVertex
		{
			public Vector3 v;
			public int color;
			public static readonly VertexFormats Format =  VertexFormats.Position | VertexFormats.Diffuse;
		};

		private int m_nBaseParticle = 0;
		private int m_nFlush = 0;
		private int m_nDiscard = 0;

		private int m_nNumParticles = 0;
		private int m_nParticlesLimit = 2048;
		private System.Collections.ArrayList m_oParticlesList = new System.Collections.ArrayList();
		private System.Collections.ArrayList freeParticles = new System.Collections.ArrayList();

		private System.Random rand = new System.Random();

		// Geometry
		private VertexBuffer m_oVertexBuffer = null;

		//11/20/2004 Chris Hill  This tracks weather or not we are in debug mode.  Note that
		//this is different than compile-debug, this is weather or not we write out a log.
		private bool m_bDebugMode = false;
		private string m_sDebugPath = "";

		private DSParticleSystem m_oParticleSystem = null;

		private string m_sParticleTexture = "";

		private System.Drawing.Color[] m_oaColors = null;
		private double[] m_naPercTimeInColor = null;
		#endregion


		public DSParticleManager()
		{
			Constructor( null,
				new System.Drawing.Color[]{ System.Drawing.Color.White,System.Drawing.Color.Black },
				new double[]{ .5,.5 },
				1,10,
				new Vector2( 0,0 ),new Vector2( 1,0 ) );
		}
		public DSParticleManager( Device oDevice )
		{
			Constructor( oDevice,
				new System.Drawing.Color[]{ System.Drawing.Color.White,System.Drawing.Color.Black },
				new double[]{ .5,.5 },
				1,10,
				new Vector2( 0,0 ),new Vector2( 1,0 ) );
		}
		public DSParticleManager( Device oDevice,
								  System.Drawing.Color[] oaColors,double[] naPercTimeInColor,
								  float fEmitVel,double nNumParticlesToEmit, 
								  Vector2 vEmitterPosition,Vector2 vEmitterDirection )
		{
			Constructor( oDevice,oaColors,naPercTimeInColor,fEmitVel,nNumParticlesToEmit,vEmitterPosition,vEmitterDirection );
		}
		private void Constructor( Device oDevice,
								  System.Drawing.Color[] oaColors,double[] naPercTimeInColor,
								  float fEmitVel,double nNumParticlesToEmit, 
								  Vector2 vEmitterPosition,Vector2 vEmitterDirection )
		{
			m_nFlush			= 512;
			m_nDiscard			= 2048;
			m_nBaseParticle		= m_nDiscard;

			m_nNumParticles		= 0;
			m_sGUID = DSMisc.GetGUID( "ParticleManager" );

			m_oaColors = oaColors;
			m_naPercTimeInColor = naPercTimeInColor;
			m_fEmitVel = fEmitVel;
			m_nNumParticlesToEmit = nNumParticlesToEmit;
			m_vEmitterPosition = vEmitterPosition;
			m_vEmitterDirection = vEmitterDirection;
			m_vEmitterDirection.Normalize();


			if( oDevice != null )
			{
				RestoreDeviceObjects( oDevice );
			}
		}

		public virtual void RestoreDeviceObjects( Device oDevice )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleManager.RestoreDeviceObjects";

			try
			{
				// Create a vertex buffer for the Particle system.  The size of this buffer
				// does not relate to the number of m_nNumParticles that exist.  Rather, the
				// buffer is used as a communication channel with the device.. we fill in 
				// a bit, and tell the device to draw.  While the device is drawing, we
				// fill in the next bit using NOOVERWRITE.  We continue doing this until 
				// we run out of vertex buffer space, and are forced to DISCARD the buffer
				// and start over at the beginning.
				oDevice.RenderState.SourceBlend = Blend.One;
				oDevice.RenderState.DestinationBlend = Blend.One;

				m_oVertexBuffer = new VertexBuffer(typeof(PointVertex), m_nDiscard, oDevice,  Usage.Dynamic | Usage.WriteOnly | Usage.Points, PointVertex.Format, Pool.Default);
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		public virtual void Advance( double dElapsedTime )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleManager.Advance";
			DSParticle oParticle = null;
			double nTempParticlesToEmit = 0;
			double nTimePerParticle = 0;
			double nCurrentEmitTime = 0;
			int nParticlesToEmitThisFrame = 0;
            bool bUseSubParts = false;
            double nAvgTime = 0;
            double nElapsedSubTime = 0;
            double nTimeLeftToUse = dElapsedTime;

			try
			{
                nAvgTime = m_oParticleSystem.AverageAdvanceTime;
                bUseSubParts = (nTimeLeftToUse / nAvgTime >= 10);

                while (nTimeLeftToUse > 0)
                {
                    if (bUseSubParts == true)
                    {
                        nElapsedSubTime = DSMisc.Min(nAvgTime, nTimeLeftToUse);
                        nTimeLeftToUse -= DSMisc.Min(nAvgTime, nTimeLeftToUse);
                    }
                    else
                    {
                        nElapsedSubTime = nTimeLeftToUse;
                        nTimeLeftToUse = 0;
                    }

                    #region Advance the manager
                    //Watch our for our life span too
                    if (m_dParticleManagerLifeTimeInSeconds > 0)
                    {
                        m_dParticleManagerTimeAlive += nElapsedSubTime;
                        if (m_dParticleManagerLifeTimeInSeconds <= m_dParticleManagerTimeAlive)
                        {
                            this.RunDown = true;
                        }
                    }

                    //Advance the particle manager itself
                    this.EmitterPosition = new Vector2(
                                (float)(this.EmitterPosition.X + this.EmittersVelocity.X * nElapsedSubTime),
                                (float)(this.EmitterPosition.Y + this.EmittersVelocity.Y * nElapsedSubTime));

                    //Advance & Kill Particles
                    m_dCurrentTime += nElapsedSubTime;
                    for (int ii = m_oParticlesList.Count - 1; ii >= 0; ii--)
                    {
                        oParticle = (DSParticle)m_oParticlesList[ii];

                        //Advance the particle
                        AdvanceParticle(nElapsedSubTime, oParticle);

                        // Kill old particles
                        if (m_dCurrentTime - oParticle.m_dCreationTime > oParticle.m_dLifeTimeOfParticle
                            && m_dParticleLifeTimeInSeconds > 0)
                        {
                            KillParticle(oParticle);

                            // Kill Particle
                            freeParticles.Add(oParticle);
                            m_oParticlesList.RemoveAt(ii);
                            m_nNumParticles--;
                        }
                        else
                        {
                            m_oParticlesList[ii] = oParticle;
                        }
                    }

                    #region Emit new particles
                    //First of all how many m_nNumParticles a second do we emit?
                    nTempParticlesToEmit = m_nNumParticlesToEmit * nElapsedSubTime + m_nNumPartialParticlesEmmitedLastClick;
                    //Make sure we don't emit fractions of a particle.  Save them up for next time
                    m_nNumPartialParticlesEmmitedLastClick = nTempParticlesToEmit - Math.Floor(nTempParticlesToEmit);
                    //Finally we have our emittion count
                    nParticlesToEmitThisFrame = m_nNumParticles + (int)Math.Floor(nTempParticlesToEmit);

                    if (m_bRunDown == false && m_bEmitParticles == true && nParticlesToEmitThisFrame > 0)
                    {
                        //How long between each particle emission?
                        nTimePerParticle = nElapsedSubTime / nParticlesToEmitThisFrame;

                        //11/30/2004 Chris Hill  If we have exceeded our particle emission amount then bail.
                        while (
                                m_nNumParticles < m_nParticlesLimit && m_nNumParticles < nParticlesToEmitThisFrame &&
                                (m_nNumParticlesEmmited <= m_nMaxNumParticlesToEmit || m_nMaxNumParticlesToEmit == 0)
                             )
                        {
                            //11/30/2004 Chris Hill  Keep track of how many m_nNumParticles we emit.
                            m_nNumParticlesEmmited++;
                            if (m_nNumParticlesEmmited > m_nMaxNumParticlesToEmit && m_nMaxNumParticlesToEmit > 0)
                            {
                                this.RunDown = true;
                            }

                            if (freeParticles.Count > 0)
                            {
                                oParticle = (DSParticle)freeParticles[0];
                                freeParticles.RemoveAt(0);
                            }
                            else
                            {
                                oParticle = new DSParticle();
                            }

                            // Emit new Particle, advance our particle emmission based on the time into emmission
                            oParticle.m_vPosition = new Vector2(
                                    (float)(m_vEmitterPosition.X + m_vEmittersVelocity.X * nCurrentEmitTime),
                                    (float)(m_vEmitterPosition.Y + m_vEmittersVelocity.Y * nCurrentEmitTime));
                            oParticle.m_dCreationTime = m_dCurrentTime;
                            oParticle.m_dLifeTimeOfParticle = this.ParticleLifeTimeInSeconds;

                            #region Get Starting Velocity
                            double nAngleChange = DSMisc.GetRnd() * m_nEmitterRangeInRad - m_nEmitterRangeInRad / 2.0;
                            double nEmitterAngle = DSMath.CalculateRadAngle(0, 0, m_vEmitterDirection.X, m_vEmitterDirection.Y);

                            oParticle.m_vVelocity = new Vector2(
                                    (float)(Math.Cos(nEmitterAngle + nAngleChange) * m_fEmitVel + m_vEmittersVelocity.X),
                                    (float)(Math.Sin(nEmitterAngle + nAngleChange) * m_fEmitVel + m_vEmittersVelocity.Y));
                            #endregion

                            CreateNewParticle(oParticle);
                            m_oParticlesList.Add(oParticle);
                            m_nNumParticles++;

                            //Now advance that particle a part of a frame.  We do this so that at low FPS the 
                            //particles still look smooth.
                            AdvanceParticle(nElapsedSubTime - nCurrentEmitTime, oParticle);
                            nCurrentEmitTime += nTimePerParticle;
                        }
                    }
                    #endregion

                    if (OneShot == true)
                    {
                        RunDown = true;
                    }
                    #endregion
                }                     
            }
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public virtual void AdvanceParticle( double dElapsedTime,DSParticle oParticle )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleManager.AdvanceParticle";

			try
			{
				/*//Figure this out... the m_nNumParticles aren't acting right
				//oParticle.m_dFadeProgression -= fElapsedTime * .025f;//0.15f; //*.25
				oParticle.m_dFadeProgression -= ( dElapsedTime / m_dParticleLifeTimeInSeconds );
				if( oParticle.m_dFadeProgression < 0.0f )
				{
					oParticle.m_dFadeProgression = 0.0f;
				}*/

				oParticle.m_vPosition += oParticle.m_vVelocity * (float)dElapsedTime;
				oParticle.m_vVelocity += m_vGravity * (float)dElapsedTime;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		public virtual int GetParticleColor( DSParticle oParticle )
		{
			System.Drawing.Color oCurColor = System.Drawing.Color.Empty;
			float nTempParticleLife = 60;
			float nTimeAlive = 0;
			float nLifePerc = 0;
			float nColorPerc = 0;
			int nPercColorToUse = m_naPercTimeInColor.Length-1;


            if (oParticle.m_dLifeTimeOfParticle > 0)
			{
				nTempParticleLife = (float)oParticle.m_dLifeTimeOfParticle;
			}

			nTimeAlive = (float)( m_dCurrentTime - oParticle.m_dCreationTime ); 
			nLifePerc = nTimeAlive / nTempParticleLife;
			
			//Now find our color bar
			for( int i=0 ; i<m_naPercTimeInColor.Length ; i++ )
			{
				if( m_naPercTimeInColor[ i ] >= nLifePerc )
				{
					nPercColorToUse = i;
					break;
				}
			}

			//Did we find it?  If not use the last one.
			if( nPercColorToUse == 0 )
			{
				nColorPerc = 0;//nLifePerc / m_naPercTimeInColor[ 0 ];
				oCurColor = m_oaColors[ 0 ];
			}
			else if( nPercColorToUse == m_naPercTimeInColor.Length-1 && nLifePerc > m_naPercTimeInColor[ nPercColorToUse ] )
			{
				nColorPerc = 0;//nLifePerc / m_naPercTimeInColor[ 0 ];
				oCurColor = m_oaColors[ nPercColorToUse ];
			}
			else
			{
				nColorPerc = (float)(
										( nLifePerc - (float)m_naPercTimeInColor[ nPercColorToUse - 1 ] ) / 
										( m_naPercTimeInColor[ nPercColorToUse ] - m_naPercTimeInColor[ nPercColorToUse - 1 ] )
									);
				oCurColor = ColorOperator.Lerp( m_oaColors[ nPercColorToUse ], 
												m_oaColors[ nPercColorToUse-1 ],1 - nColorPerc );
			}


			return( oCurColor.ToArgb() );
		}
		public virtual void KillParticle( DSParticle oParticle )
		{
		}
		public virtual void CreateNewParticle( DSParticle oParticle )
		{
		}
		public virtual void Render( Device o3DDevice,Vector2 oViewUpperLeft )
		{
			const string sRoutineName = "DarkStrideToolbox.ParticleManager.Render";
			PointVertex[] vertices = null;
			int numParticlesToRender = 0;
			int count = 0;
			Vector3 vPos;
			Vector2 vVel;
			float fLengthSq;
			Texture oTexture = null;
			//System.Reflection.Assembly oAssembly = null;
			//System.IO.Stream oStream = null;

			try
			{
				//11/20/2004 Chris Hill  Added debugging.
				DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,m_bDebugMode,3,"Starting ParticleManager.Render" );


				if( m_sParticleTexture != null && m_sParticleTexture.Length > 0 )
				{
					oTexture = DSResourceManager.GetGlobalInstance().GetTexture( m_sParticleTexture );
				}
				else
				{
					oTexture = DSResourceManager.GetGlobalInstance().GetTexture( "System_Particle" );				
				}
				o3DDevice.SetTexture(0, oTexture);

				// Set up the vertex buffer to be rendered
				o3DDevice.SetStreamSource(0, m_oVertexBuffer, 0);
				o3DDevice.VertexFormat = PointVertex.Format;

				o3DDevice.RenderState.ZBufferWriteEnable = false;
				o3DDevice.RenderState.AlphaBlendEnable = true;
				o3DDevice.RenderState.SourceBlend = Blend.One;
				o3DDevice.RenderState.DestinationBlend = Blend.One;

				//Set the render states for using point sprites
				o3DDevice.RenderState.PointSpriteEnable = true;
				o3DDevice.RenderState.PointScaleEnable = false;
				o3DDevice.RenderState.PointSize = m_fParticleSize;
				o3DDevice.RenderState.PointSizeMin = 0.00f;
				o3DDevice.RenderState.PointScaleA = 0.00f;
				o3DDevice.RenderState.PointScaleB = 0.00f;
				o3DDevice.RenderState.PointScaleC = 1.00f;

				o3DDevice.SetTextureStageState( 0, TextureStageStates.ColorOperation, (int)TextureOperation.Modulate ); // Modulate...
				o3DDevice.SetTextureStageState( 0, TextureStageStates.ColorArgument1, (int)TextureArgument.TextureColor );   // the texture for this stage with...
				o3DDevice.SetTextureStageState( 0, TextureStageStates.ColorArgument2, (int)TextureArgument.Diffuse );   // the diffuse color of the geometry.
				o3DDevice.SetTextureStageState( 0, TextureStageStates.AlphaOperation, (int)TextureOperation.SelectArg1 );   
				o3DDevice.SetTextureStageState( 0, TextureStageStates.AlphaArgument1, (int)TextureArgument.TextureColor );
				o3DDevice.SetTextureStageState( 1, TextureStageStates.ColorOperation, (int)TextureOperation.Disable );
				o3DDevice.SetTextureStageState( 1, TextureStageStates.AlphaOperation, (int)TextureOperation.Disable );



				// Lock the vertex buffer.  We fill the vertex buffer in small
				// chunks, using LockFlags.NoOverWrite.  When we are done filling
				// each chunk, we call DrawPrim, and lock the next chunk.  When
				// we run out of space in the vertex buffer, we start over at
				// the beginning, using LockFlags.Discard.
				m_nBaseParticle += m_nFlush;

				if (m_nBaseParticle >= m_nDiscard)
					m_nBaseParticle = 0;

				vertices = (PointVertex[])m_oVertexBuffer.Lock(m_nBaseParticle * DXHelp.GetTypeSize(typeof(PointVertex)), typeof(PointVertex), (m_nBaseParticle != 0) ? LockFlags.NoOverwrite : LockFlags.Discard, m_nFlush);
				foreach( DSParticle oLoopParticle in m_oParticlesList )
				{
					//11/19/2004 Chris Hill  This code was assuming the viewport was 500x500!!
					vPos = new Vector3( 
									( -( oViewUpperLeft.X + ( o3DDevice.Viewport.Width  / 2.0f ) ) + oLoopParticle.m_vPosition.X ) / ( o3DDevice.Viewport.Width  / 2.0f ), 
									(  ( oViewUpperLeft.Y + ( o3DDevice.Viewport.Height / 2.0f ) ) - oLoopParticle.m_vPosition.Y ) / ( o3DDevice.Viewport.Height / 2.0f ),
									0 );

					vVel = oLoopParticle.m_vVelocity;
					fLengthSq = vVel.LengthSq();

					vertices[count].v     = vPos;
					vertices[count].color = GetParticleColor( oLoopParticle );
					count++;

					if( ++numParticlesToRender == m_nFlush )
					{
						//11/20/2004 Chris Hill  Added debugging.
						DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,m_bDebugMode,4,"Flush the buffer" );


						// Done filling this chunk of the vertex buffer.  Lets unlock and
						// draw this portion so we can begin filling the next chunk.
						m_oVertexBuffer.Unlock();
						o3DDevice.DrawPrimitives(PrimitiveType.PointList, m_nBaseParticle, numParticlesToRender);

						// Lock the next chunk of the vertex buffer.  If we are at the 
						// end of the vertex buffer, LockFlags.Discard the vertex buffer and start
						// at the beginning.  Otherwise, specify LockFlags.NoOverWrite, so we can
						// continue filling the VB while the previous chunk is drawing.
						m_nBaseParticle += m_nFlush;

						if( m_nBaseParticle >= m_nDiscard )
						{
							m_nBaseParticle = 0;
						}

						vertices = (PointVertex[])m_oVertexBuffer.Lock(m_nBaseParticle * DXHelp.GetTypeSize(typeof(PointVertex)), typeof(PointVertex), (m_nBaseParticle != 0) ? LockFlags.NoOverwrite : LockFlags.Discard, m_nFlush);
						count = 0;

						numParticlesToRender = 0;
					}
				} 

				// Unlock the vertex buffer
				m_oVertexBuffer.Unlock();
				// Render any remaining m_nNumParticles
				if (numParticlesToRender > 0)
					o3DDevice.DrawPrimitives(PrimitiveType.PointList, m_nBaseParticle, numParticlesToRender);


				// Reset render states
				o3DDevice.RenderState.PointSpriteEnable = false;
				o3DDevice.RenderState.PointScaleEnable = false;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}


		#region Properties
		public long ParticleCount
		{
			get
			{
				return( m_oParticlesList.Count );
			}
		}
		//Movement Properties
		public Vector2 Gravity
		{
			get
			{
				return( m_vGravity );
			}
			set
			{
				m_vGravity = value;
			}
		}

		public Vector2 EmitterPosition
		{
			get
			{
				return( m_vEmitterPosition );
			}
			set
			{
				m_vEmitterPosition = value;
			}
		}
		public Vector2 EmitterDirection
		{
			get
			{
				return( m_vEmitterDirection );
			}
			set
			{
				m_vEmitterDirection = value;
			}
		}

		public Vector2 EmittersVelocity
		{
			get
			{
				return( m_vEmittersVelocity );
			}
			set
			{
				m_vEmittersVelocity = value;
			}
		}
		public double EmitterRangeInRad
		{
			get
			{
				return( m_nEmitterRangeInRad );
			}
			set
			{
				m_nEmitterRangeInRad = value;
			}
		}
		//Particle Properties
		public double ParticleLifeTimeInSeconds
		{
			get
			{
				return( m_dParticleLifeTimeInSeconds );
			}
			set
			{
				m_dParticleLifeTimeInSeconds = value;
			}
		}
        public double ParticleManagerLifeTimeInSeconds
        {
            get
            {
                return (m_dParticleManagerLifeTimeInSeconds);
            }
            set
            {
                m_dParticleManagerLifeTimeInSeconds = value;
            }
        }

		//General Properties
		public virtual string GUID
		{
			get
			{
				return( m_sGUID );
			}
			set
			{
				m_sGUID = value;
			}
		}
		//Apperance Properties
		public float ParticleSize
		{
			get
			{
				return( m_fParticleSize );
			}
			set
			{
				m_fParticleSize = value;
			}
		}
		public bool RunDown
		{
			get
			{
				return( m_bRunDown );
			}
			set
			{
				m_bRunDown = value;
			}
		}
		public bool OneShot
		{
			get
			{
				return( m_bOneShot );
			}
			set
			{
				m_bOneShot = value;
			}
		}
		public bool EmitParticles
		{
			get
			{
				return( m_bEmitParticles );
			}
			set
			{
				m_bEmitParticles = value;
			}
		}
		public float EmitVelocity
		{
			get
			{
				return( m_fEmitVel );
			}
			set
			{
				m_fEmitVel = value;
			}
		}

		//11/20/2004 Chris Hill  Added a debug mode for tracking a time problem.
		public bool DebugMode
		{
			set
			{
				//Are we turning it on when it was off?
				if( m_bDebugMode == false && value == true )
				{
					//Only make a new log file name if we don't have one already
					if( m_sDebugPath.Length == 0 )
					{
						m_sDebugPath =	DSMisc.GetDevelopmentAppPath() + 
							"DSParticleMgr_DebugFile_" + 
							DateTime.Now.ToString( "yyyy-MM-dd hh" ) + "h" + 
							DateTime.Now.ToString( "mm" ) + "m" + 
							DateTime.Now.ToString( "ss" ) + "s" + 
							".OUT";

						//Then write a header
						DSBufferDebug.Inst().WriteToDebug(  m_sDebugPath,value,false,0,
							"\r\n\r\nStarting Debug Log File For DSParticleManager\r\n" +
							"--------------------------------------------" );
					}

				}
					//Are we turning it off when it was currently on?  Then clear the path.
				else if( m_bDebugMode == true && value == false )
				{
					m_sDebugPath = "";
				}

				m_bDebugMode = value;
			}
			get
			{
				return( m_bDebugMode );
			}
		}
		public string DebugPath
		{
			get
			{
				return( m_sDebugPath );
			}
			set
			{
				m_sDebugPath = value;

				//We turn off then on the debug mode so that it writes a new header to the log
				if( m_bDebugMode == true )
				{
					DebugMode = false;
					DebugMode = true;
				}				
			}
		}
		public double CurrentTime
		{
			get
			{
				return( m_dCurrentTime );
			}
		}
		public double NumParticlesToEmit
		{
			set
			{
				m_nNumParticlesToEmit = value;
			}
			get
			{
				return( m_nNumParticlesToEmit );
			}
		}
		public DSParticleSystem ParticleSystem
		{
			get
			{
				return( m_oParticleSystem );
			}
			set
			{
				m_oParticleSystem = value;
			}
		}
		//11/30/2004 Chris Hill  This is the number of m_nNumParticles to emit before we shut down.
		public long MaxNumParticlesToEmit
		{
			get
			{
				return( m_nMaxNumParticlesToEmit );
			}
			set
			{
				m_nMaxNumParticlesToEmit = value;
			}
		}
		//11/30/2004 Chris Hill  This is the max number of m_nNumParticles we can have in existance at any one time.
		public int ParticlesLimit
		{
			get
			{
				return( m_nParticlesLimit );
			}
			set
			{
				m_nParticlesLimit = value;
			}
		}
		public string ParticleTexture
		{
			get
			{
				return( m_sParticleTexture );
			}
			set
			{
				m_sParticleTexture = value;
			}
		}
		public System.Drawing.Color[] Colors
		{
			get
			{
				return( m_oaColors );
			}
			set
			{
				m_oaColors = value;
			}
		}
		public double[] PercTimeInColor
		{
			get
			{
				return( m_naPercTimeInColor );
			}
			set
			{
				m_naPercTimeInColor = value;

                for (int i = 1; i < m_naPercTimeInColor.Length; i++)
                {
                    if (m_naPercTimeInColor[i-1] > m_naPercTimeInColor[i])
                    {
                        throw new System.Exception("The percent time is cumulative, each percent time field should be the previous value plus some new percent.");
                    }
                }
			}
		}
        protected int BaseParticle
		{
			get
			{
				return( m_nBaseParticle );
			}
			set
			{
				m_nBaseParticle = value;
			}
		}
		protected VertexBuffer VertexBuffer
		{
			get
			{
				return( m_oVertexBuffer );
			}
			set
			{
				m_oVertexBuffer = value;
			}
		}

		protected int Flush
		{
			get
			{
				return( m_nFlush );
			}
			set
			{
				m_nFlush = value;
			}
		}
		protected int Discard
		{
			get
			{
				return( m_nDiscard );
			}
			set
			{
				m_nDiscard = value;
			}
		}
		protected System.Collections.ArrayList ParticlesList
		{
			get
			{
				return( m_oParticlesList );
			}
			set
			{
				m_oParticlesList = value;
			}
		}
		#endregion
	}


	public class DSParticle
	{
		public Vector2 m_vPosition;         // Current position
		public Vector2 m_vVelocity;         // Current velocity
		public double m_dCreationTime;      // Time of creation
        public double m_dLifeTimeOfParticle = 0;
	};
}